from typing import Literal
import ext.aflow.scripts.optimized.MBPP.workflows.template.operator as operator
import ext.aflow.scripts.optimized.MBPP.workflows.round_20.prompt as prompt_custom
from provider.llm_provider_registry import create_llm_instance
from utils.cost_manager import CostManager

DatasetType = Literal["HumanEval", "MBPP", "GSM8K", "MATH", "HotpotQA", "DROP"]

class Workflow:
    def __init__(
        self,
        name: str,
        llm_config,
        dataset: DatasetType,
    ) -> None:
        self.name = name
        self.dataset = dataset
        self.llm = create_llm_instance(llm_config)
        self.llm.cost_manager = CostManager()
        self.custom = operator.Custom(self.llm)
        self.custom_code_generate = operator.CustomCodeGenerate(self.llm)
        self.sc_ensemble = operator.ScEnsemble(self.llm)
        self.test = operator.Test(self.llm)

    async def __call__(self, problem: str, entry_point: str):
        # Analyze the problem requirements
        problem_analysis = await self.custom(input=problem, instruction=prompt_custom.PROBLEM_ANALYSIS_PROMPT)
        
        # Generate test cases
        test_cases = await self.custom(input=f"Problem: {problem}\nProblem Analysis: {problem_analysis['response']}", instruction=prompt_custom.GENERATE_TEST_CASES_PROMPT)
        
        # Generate multiple solutions
        solutions = []
        for _ in range(3):
            solution = await self.custom_code_generate(problem=problem, entry_point=entry_point, instruction=prompt_custom.CODE_GENERATE_PROMPT)
            solutions.append(solution['response'])
        
        # Use ScEnsemble to select the best solution
        best_solution = await self.sc_ensemble(solutions=solutions, problem=problem)
        
        # Test the selected solution
        test_result = await self.test(problem=problem, solution=best_solution['response'], entry_point=entry_point)
        
        if test_result['result']:
            return test_result['solution'], self.llm.cost_manager.total_cost
        else:
            # If the test fails, analyze the error
            error_analysis = await self.custom(input=f"Problem: {problem}\nProblem Analysis: {problem_analysis['response']}\nTest Cases: {test_cases['response']}\nFailed solution: {best_solution['response']}\nError: {test_result['solution']}", instruction=prompt_custom.ERROR_ANALYSIS_PROMPT)
            
            # Generate multiple improved solutions
            improved_solutions = []
            for _ in range(3):
                improved_solution = await self.custom(input=f"Problem: {problem}\nProblem Analysis: {problem_analysis['response']}\nTest Cases: {test_cases['response']}\nFailed solution: {best_solution['response']}\nError: {test_result['solution']}\nError analysis: {error_analysis['response']}", instruction=prompt_custom.IMPROVE_SOLUTION_PROMPT)
                improved_solutions.append(improved_solution['response'])
            
            # Use ScEnsemble to select the best improved solution
            best_improved_solution = await self.sc_ensemble(solutions=improved_solutions, problem=problem)
            
            return best_improved_solution['response'], self.llm.cost_manager.total_cost
